// utils/pagination.js
class Pagination {
  constructor(options = {}) {
    this.container = options.container || "#pagination";
    this.pageLengthSelector = options.pageLengthSelector || "#pageLength";
    this.tableBody = options.tableBody || "tbody";
    this.totalRowsElement = options.totalRowsElement || "#totalRows";
    this.defaultLimit = options.defaultLimit || 10;
    this.currentPage = 1;
    this.currentLimit = this.defaultLimit;
    this.currentSort = options.defaultSort || { column: "id", order: "DESC" };
    this.searchTerm = "";
    this.filters = {};
    this.dataMapper = options.dataMapper || ((item) => item);
    this.fetchUrl = options.fetchUrl || "";
    this.initialize();
  }

  initialize() {
    // Set up event listeners
    $(this.pageLengthSelector).change(() => {
      this.currentLimit = $(this.pageLengthSelector).val();
      this.loadData(1);
    });

    if (!$(this.container).length) {
      console.error(`Pagination container not found: ${this.container}`);
    }
    if (!$(this.pageLengthSelector).length) {
      console.error(
        `Page length selector not found: ${this.pageLengthSelector}`
      );
    }

    if (typeof window.globalSearchInput !== "undefined") {
      $(window.globalSearchInput).on("input", () => {
        // Clear filters when using global search
        this.filters = {};
        this.searchTerm = $(window.globalSearchInput).val().trim();
        this.loadData(1);
      });
    }
  }

  async loadData(page = 1, customFilters = null) {  // Changed to null as default
    this.currentPage = page;
    
    // If customFilters is provided (not null), we're applying filters
    if (customFilters !== null) {
      this.searchTerm = '';
      this.filters = customFilters;
      if (window.globalSearchInput) {
        $(window.globalSearchInput).val('');
      }
    }
    
    const queryParams = new URLSearchParams({
      page: this.currentPage,
      limit: this.currentLimit,
      sortBy: this.currentSort.column,
      sortOrder: this.currentSort.order
    });

    // Explicitly use either search or filters, never both
    if (this.searchTerm) {
      queryParams.append('search', this.searchTerm);
    } else {
      // Add filters only if they exist and we're not searching
      for (const [key, value] of Object.entries(this.filters)) {
        if (value) {
          queryParams.append(key, value);
        }
      }
    }

    try {

 
      const response = await $.get(`${this.fetchUrl}?${queryParams.toString()}`);
  
      // console.log("API Response:", response);
  
      const data = response.data || response.trips || response.items || [];
      const totalItems = response.totalItems || response.totalUsers || 0;
      const totalPages = response.totalPages || Math.ceil(totalItems / this.currentLimit);
  
      if (!data.length) {
        $(this.tableBody).html('<tr><td colspan="100%">No data found</td></tr>');
            $(this.totalRowsElement).text(0);

        return;
      }
  
      this.renderTable(data);
      this.renderPagination(totalPages, totalItems);
  
      if (window.exportConfig) {
        window.exportConfig.currentData = data;
      }
  
      return response;
    } catch (error) {
      console.error("Error fetching data:", error);
    }
  }
  

  renderTable(data) {
    let tbody = "";
    data.forEach((item) => {
      tbody += this.dataMapper(item);
    });
    $(this.tableBody).html(tbody);
  }

  renderPagination(totalPages, totalItems) {
    let paginationHtml = `<nav aria-label="Page navigation"><ul class="pagination justify-content-center">`;

    // First Page Button
    if (this.currentPage > 1) {
      paginationHtml += `<li class="page-item">
            <a class="page-link" href="javascript:void(0)" onclick="event.preventDefault(); pagination.loadData(1)">First</a>
          </li>`;
    }

    // Previous Page Button
    if (this.currentPage > 1) {
      paginationHtml += `<li class="page-item">
            <a class="page-link" href="javascript:void(0)" onclick="event.preventDefault(); pagination.loadData(${
              this.currentPage - 1
            })">&laquo;</a>
          </li>`;
    }

    // Page Numbers
    let startPage = Math.max(1, this.currentPage - 2);
    let endPage = Math.min(totalPages, startPage + 4);

    for (let i = startPage; i <= endPage; i++) {
      paginationHtml += `
            <li class="page-item ${i === this.currentPage ? "active" : ""}">
              <a class="page-link" href="javascript:void(0)" onclick="event.preventDefault(); pagination.loadData(${i})">${i}</a>
            </li>`;
    }

    // Next Page Button
    if (this.currentPage < totalPages) {
      paginationHtml += `<li class="page-item">
            <a class="page-link" href="javascript:void(0)" onclick="event.preventDefault(); pagination.loadData(${
              this.currentPage + 1
            })">&raquo;</a>
          </li>`;
    }

    // Last Page Button
    if (this.currentPage < totalPages) {
      paginationHtml += `<li class="page-item">
            <a class="page-link" href="javascript:void(0)" onclick="event.preventDefault(); pagination.loadData(${totalPages})">Last</a>
          </li>`;
    }

    paginationHtml += `</ul></nav>`;

    $(this.container).html(paginationHtml);
    $(this.totalRowsElement).text(totalItems || 0);
  }

  setSort(column) {
    const newOrder =
      this.currentSort.column === column && this.currentSort.order === "asc"
        ? "desc"
        : "asc";
    this.currentSort = { column, order: newOrder };
    this.loadData(1);
  }
}

// Attach event listener once after DOM loads
document.addEventListener("DOMContentLoaded", function () {
  document.querySelector("table").addEventListener("click", function (e) {
    const th = e.target.closest("th[data-sort]");
    if (th) {
      const column = th.getAttribute("data-sort");
      pagination.setSort(column);
    }
  });
});


// Export for Node.js or assign to window for browser
if (typeof module !== "undefined" && module.exports) {
  module.exports = Pagination;
} else {
  window.Pagination = Pagination;
}
