document.addEventListener("DOMContentLoaded", () => {
  const form = document.getElementById("editUserForm");
  if (!form) return;

  const phone = document.getElementById("phone").value.trim();
  const email = document.getElementById("email").value.trim();

  // Validate phone
  if (!/^\d{6,15}$/.test(phone)) {
    return Swal.fire(
      "Invalid Phone Number",
      "Phone number must be between 6 to 15 digits.",
      "error"
    );
  }

  // Validate email only if not empty
  if (email !== "") {
    const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailPattern.test(email)) {
      return Swal.fire(
        "Invalid Email",
        "Please enter a valid email address.",
        "error"
      );
    }
  }

  const formValidator = setupFormValidation(
    formConfig.formId,
    formConfig.fields
  );

  form.addEventListener("submit", async function (e) {
    e.preventDefault();
    if (!formValidator.validateForm()) return;

    const fileInput = document.getElementById("profilePic");
    const file = fileInput.files[0];
    const api_key = document.getElementById("api_key").value;
    const user_id = document.getElementById("user_id").value;
    const city_id = document.getElementById("city_id").value;

    const currentValues = {
      u_fname: document.getElementById("u_fname").value,
      u_lname: document.getElementById("u_lname").value,
      u_email: document.getElementById("email").value.trim(),
      u_phone: document.getElementById("phone").value.trim(),
      c_code: document.getElementById("countryCode").value,
      active: document.getElementById("status").value,
      city_id: document.getElementById("city_id").value,
      profilePic: file ? file.name : "",
    };

  

    let payload = {
      user_id,
      is_return_details: 1,
      api_key,
      city_id
    };

    let changed = false;

    for (const key in currentValues) {
      if (currentValues[key] !== initialValues[key]) {
        // Skip profilePic here; it's handled separately
        if (key === "u_email" && currentValues[key] === "") continue;
        if (key !== "profilePic") {
          payload[key] = currentValues[key];
        }
        changed = true;
      }
    }

    // If no change in text fields and no new image uploaded
    if (!changed && !file) {
      return Swal.fire({
        icon: "warning",
        title: "No changes detected",
        toast: true,
        position: "top-end",
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true,
      });
    }

    // Construct u_name if fname or lname changed
    if (payload.u_fname || payload.u_lname) {
      const fname = payload.u_fname || initialValues.u_fname;
      const lname = payload.u_lname || initialValues.u_lname;
      payload.u_name = `${fname} ${lname}`;
    }

    if (file) {
      const reader = new FileReader();
      reader.onloadend = async function () {
        payload.user_image = reader.result.split(",")[1];
        payload.image_type = file.type.split("/")[1];
        await sendUpdateRequest(payload, api_key, currentValues, user_id);
      };
      reader.readAsDataURL(file);
    } else {
      await sendUpdateRequest(payload, api_key, currentValues, user_id);
    }
  });

  async function sendUpdateRequest(payload, api_key, currentValues, user_id) {
    try {
      document.getElementById("loaderOverlay").style.display = "flex";

      const bodyParams = new URLSearchParams();
      bodyParams.append("c_code", currentValues.c_code);
      bodyParams.append("u_phone", currentValues.u_phone);
      if (currentValues.u_email) {
        bodyParams.append("u_email", currentValues.u_email);
      }

      const validateResponse = await fetch(`/api/v1/userapi/uservalidate`, {
        method: "POST",
        headers: { "Content-Type": "application/x-www-form-urlencoded" },
        body: bodyParams,
      });

      const validateResult = await validateResponse.json();

      if (
        validateResult.resDecrypt?.response?.user_id != user_id &&
        validateResult.resDecrypt?.response !== true
      ) {
        document.getElementById("loaderOverlay").style.display = "none";
        return Swal.fire(
          "Validation Failed",
          validateResult.message || "Email or phone already exists",
          "error"
        );
      }

      const response = await fetch(
        `/api/v1/userapi/updateuserprofile?api_key=${api_key}`,
        {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify(payload),
        }
      );

      const result = await response.json();
      handleResponse(result);
    } catch (err) {
      document.getElementById("loaderOverlay").style.display = "none";
      Swal.fire({
        icon: "error",
        title: "Request Failed",
        text: err.message || "Unknown error occurred.",
      });
    }
  }

  function handleResponse(result) {
    document.getElementById("loaderOverlay").style.display = "none";
    if (result.resDecrypt?.code === 200) {
      Swal.fire({
        icon: "success",
        title: "Passenger updated successfully",
        toast: true,
        position: "top-end",
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true,
      });
    } else {
      Swal.fire({
        icon: "error",
        title: "Update Failed",
        text: result.message || "Something went wrong.",
        toast: true,
        position: "top-end",
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true,
      });
    }
  }
});
