// formValidation.js
function setupFormValidation(formId, fieldConfigs) {
  const form = document.getElementById(formId);
  if (!form) {
    console.error(`Form with ID ${formId} not found`);
    return null;
  }

  // Add CSS for error states if not already present
  if (!document.querySelector("style[data-form-validation]")) {
    const style = document.createElement("style");
    style.dataset.formValidation = "true";
    style.textContent = `
      .is-invalid {
        border-color: #dc3545 !important;
      }
      .text-danger {
        color: #dc3545;
        font-size: 0.875em;
      }
    `;
    document.head.appendChild(style);
  }

  // Add error containers for each field
  fieldConfigs.forEach((config) => {
    const field = form.querySelector(`[name="${config.name}"]`);
    if (field) {
      // Create error container if it doesn't exist
      let errorContainer = field.parentNode.querySelector(
        `#${config.name}-error`
      );
      if (!errorContainer) {
        errorContainer = document.createElement("div");
        errorContainer.className = "text-danger mt-1";
        errorContainer.id = `${config.name}-error`;
        field.parentNode.appendChild(errorContainer);
      }

      // Add event listeners for real-time validation
      field.addEventListener("blur", () => validateField(field, config));
    }
  });

  // Field validation function
  function validateField(field, config) {
    const value = field.value.trim();
    const errorContainer = document.getElementById(`${config.name}-error`);

    if (!errorContainer) return true;

    // Clear previous error
    errorContainer.textContent = "";
    field.classList.remove("is-invalid");

    // Required validation
   // ✅ Required validation with support for existing image URL
  if (config.required && !value) {
    if (config.type === "file" && config.existingUrl) {
      const url = config.existingUrl.trim();
      if (url && !url.includes("default") && !url.startsWith("data:")) {
        return true; // Existing URL is valid
      }
    }

    showError(
      field,
      errorContainer,
      config.messages?.required || "This field is required"
    );
    return false;
  }
    // Type-specific validation
    if (config.type === "email" && value) {
      const emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;

      if (!emailRegex.test(value.trim())) {
        showError(
          field,
          errorContainer,
          config.messages?.invalid || "Please enter a valid email address."
        );
        return false;
      }
    }

    if (config.type === "phone" && value) {
      const phoneRegex = /^\d{6,15}$/;
      if (!phoneRegex.test(value)) {
        showError(
          field,
          errorContainer,
          config.messages?.invalid || "Phone number must be 6-15 digits"
        );
        return false;
      }
    }

    if (config.type === "select" && config.required && value === "") {
      showError(
        field,
        errorContainer,
        config.messages?.required || "Please select an option"
      );
      return false;
    }

    // Custom validation if provided
    if (config.customValidation) {
      const customError = config.customValidation(value);
      if (customError) {
        showError(field, errorContainer, customError);
        return false;
      }
    }
 

    return true;
  }

  function showError(field, errorContainer, message) {
    errorContainer.textContent = message;
    field.classList.add("is-invalid");
  }

  return {
    validateForm: function () {
      let isValid = true;
      fieldConfigs.forEach((config) => {
        const field = form.querySelector(`[name="${config.name}"]`);
        if (field && !validateField(field, config)) {
          isValid = false;
        }
      });
      return isValid;
    },
  };
}

function validateImage(inputId, previewId, errorId) {
  let MAX_SIZE = 100 * 1024; // 100KB in bytes

  const input = document.getElementById(inputId);
  const preview = document.getElementById(previewId);
  const error = document.getElementById(errorId);

  input.addEventListener("change", function () {
    const file = this.files[0];
    error.textContent = "";
    preview.style.display = "none";

    if (file) {
      if (file.size > MAX_SIZE) {
        error.textContent = "Image must be less than 100KB.";
        input.value = "";
      } else {
        const reader = new FileReader();
        reader.onload = function (e) {
          preview.src = e.target.result;
          preview.style.display = "block";
          preview.style.maxWidth = "200px";
        };
        reader.readAsDataURL(file);
      }
    }
  });
}
